/* $Id: DBI.xs,v 1.26 1994/10/28 14:24:13 timbo Exp timbo $
 *
 * Copyright (c) 1994, Tim Bunce
 *
 * You may distribute under the terms of either the GNU General Public
 * License or the Artistic License, as specified in the Perl README file.
 */

#define IN_DBI_XS 1		/* see DBIXS.h */

#include "DBIXS.h"		/* DBI public interface for DBD's written in C	*/


#define DBI_MAGIC '~'
extern int perl_destruct_level;

static dbih_com_t *dbih_getcom _((SV *h));

static SV *dbih_event _((dbih_com_t *com, char *name, SV *a1, SV *a2));
static SV *dbi_last_h;

DBISTATE_DECLARE;


/* Internal Method Attributes (attached to dispatch methods when installed)	*/
#define IMA_MINARGS	0
#define IMA_MAXARGS	1
#define IMA_ARGS	2
#define IMA_MAX 	3	/* must me one more than largest IMA_* value */
#define IMA_FETCH(av,attrib)  (*av_fetch((av), (attrib), 1))



static void
dbi_bootinit()
{
    Newz(1, dbis, sizeof(*dbis), dbistate_t);
    /* store version and size so we can spot DBI/DBD version mismatch	*/
    dbis->version = DBISTATE_VERSION;
    dbis->size    = sizeof(*dbis);
    /* publish address of dbistate so dynaloaded DBD's can find it	*/
    sv_setiv(perl_get_sv(DBISTATE_PERLNAME,1), (IV)dbis);

    DBISTATE_INIT; /* check DBD code to set dbis from DBISTATE_PERLNAME	*/

    dbis->log	= stderr;
    dbis->debug	= 0;
    dbis->debugpvlen = 200;
    /* store some function pointers so DBD's can call our functions	*/
    dbis->getcom = dbih_getcom;
    dbis->event  = dbih_event;

    /* Remember the last handle used. BEWARE! Sneaky stuff here!	*/
    /* We want a handle reference but we don't want to increment	*/
    /* the handles reference count and we don't want perl to try	*/
    /* to destroy it during global destruction. */
    dbi_last_h  = newRV(&sv_undef);
    SvROK_off(dbi_last_h);	/* so sv_clean_objs() won't destroy it	*/
}


/* ----------------------------------------------------------------- */
/* Utility functions                                                 */


static char *
neatsvpv(sv)	/*print a tidy ascii value, for debugging only */
    SV * sv;
{
    STRLEN len;
    char *v= (SvOK(sv)) ? SvPV(sv,len) : "undef";
    SV *nsv = NULL;
    /* undef and numbers get no special treatment */
    if (!SvOK(sv) || SvIOK(sv) || SvNOK(sv))
	return v;
    /* In the future refs with overload magic will need handling */
    /* because the '""' method may return long or binary strings */
    if (SvROK(sv))
	return v;
    /* for strings we limit the length and translate codes */
    nsv = sv_2mortal(newSVpv("'",1));
    if (len > dbis->debugpvlen){
	sv_catpvn(nsv, v, dbis->debugpvlen);
	sv_catpv( nsv, "...");
    }else{
	sv_catpvn(nsv, v, len);
	sv_catpv( nsv, "'");
    }
    v = SvPV(nsv, len);
    while(len-- > 0){ /* cleanup string (map control chars to ascii etc) */
	if (!isprint(v[len]))
	    v[len] = '?';
    }
    return v;
}


static SV *
dbih_inner(orv, what)	/* convert outer to inner handle else croak */
    SV *orv;         	/* ref of outer hash */
    char *what;		/* error msg, NULL=no croak and return NULL */
{
    MAGIC *mg;
    SV *hrv;
    if (!SvROK(orv) || SvTYPE(SvRV(orv)) != SVt_PVHV){
	if (!what)
	    return NULL;
	croak("%s handle '%s' is not a hash reference",
		what, SvPV(orv,na));
    }
    if (!SvMAGICAL(SvRV(orv)) || (mg=mg_find(SvRV(orv),'P')) == NULL){
	/* maybe it's already an inner handle... */
	if (mg_find(SvRV(orv), DBI_MAGIC) == NULL){
	    if (!what)
		return NULL;
	    croak("%s handle '%s' is not a DBI handle",
		    what, SvPV(orv,na));
	}
	hrv = orv; /* was already a DBI handle inner hash */
    }else{
	hrv = mg->mg_obj;  /* inner hash of tie */
    }

    /* extra checks if being paranoid */
    if (dbis->debug && (!SvROK(hrv) || SvTYPE(SvRV(hrv)) != SVt_PVHV)){
	if (!what)
	    return NULL;
	croak("panic: %s inner handle '%s' is not a hash ref",
		what, SvPV(hrv,na));
    }
    return hrv;
}


/* ----------------------------------------------------------------- */
/* Functions to manage a DBI handle (magic and attributes etc).      */

static dbih_com_t *
dbih_getcom(hrv)
    SV *hrv;
{
    MAGIC *mg;
    AV *av;
    if ( (mg=mg_find(SvRV(hrv),  DBI_MAGIC)) == NULL){
	/* Umm, maybe it's an outer handle */
	hrv = dbih_inner(hrv, "dbih_iha");
	mg=mg_find(SvRV(hrv), DBI_MAGIC);
    }
    av = (AV*)SvRV(mg->mg_obj);
    return (dbih_com_t*)(SvPVX(*av_fetch(av, 0, 1)));
}


static char *
mkvname(stash, item, uplevel)
    HV *stash;
    char *item;
    int uplevel;
{
    SV *sv = sv_newmortal();
    sv_setpv(sv, HvNAME(stash));
    if(uplevel){
	while(SvCUR(sv) && *SvEND(sv)!=':')
	    --SvCUR(sv);
	if (SvCUR(sv))
	    --SvCUR(sv);
    }
    sv_catpv(sv, "::");
    sv_catpv(sv, item);
    SvPV(sv, na);
}


static SV *
dbih_setup_attrib(h, attrib, parent)
    SV *h;
    char *attrib;
    SV *parent;
{
    int len = strlen(attrib);
    SV *asv = *hv_fetch((HV*)SvRV(h), attrib, len, 1);
    SV *psv;
    if (SvOK(asv))	/* attribute already exists */
	return asv;
    if (!parent || !SvTRUE(parent))
	croak("dbih_setup_attrib(%s): '%s' not set and no parent supplied",
		SvPV(h,na), attrib);
    psv = *hv_fetch((HV*)SvRV(parent), attrib, len, 0);
    if (!SvOK(psv)){	/* not defined in parent */
	croak("dbih_setup_attrib(%s): '%s' not set and not in parent",
		SvPV(h,na), attrib);
    }
    sv_setsv(asv, psv); /* copy attribute from parent to handle */
    return asv;
}


static void
dbih_setup_handle(orv, imp_class, parent, imp_datasv)
    SV *orv;         /* ref of outer hash */
    char *imp_class;
    SV *parent;
    SV *imp_datasv;
{
    SV *h;
    HV *imp_stash;
    char *errmsg = "Can't dbih_setup_handle of %s to %s: %s";
    SV *comsv;
    SV *s[99];
    int i;
    dbih_com_t c;

    if ( (imp_stash = gv_stashpv(imp_class, FALSE)) == NULL)
        croak(errmsg, SvPV(orv,na), imp_class, "unknown package");

    h = dbih_inner(orv, "dbih_setup_handle");

    if (mg_find(SvRV(h),  DBI_MAGIC) != NULL)
	croak(errmsg, SvPV(orv,na), imp_class, "already a DBI handle");

    if (dbis->debug >= 2)
	fprintf(dbis->log,"    dbih_setup_handle(%s=>%s, %s, %lx)\n",
	    SvPV(orv,na), SvPV(h,na), imp_class, SvIV(imp_datasv));

    parent = dbih_inner(parent,NULL); /* check parent is valid inner handle */

    memzero(&c, sizeof(dbih_com_t));
    memzero(&s, sizeof(s));

#   define NRVHV_FETCH(name) newRV(dbih_setup_attrib(h, (name), parent))
    s[i=0]= NULL; /* This is assigned last, see below */
    s[++i]= c.Err_rv      = NRVHV_FETCH("Err");
    s[++i]= c.Errstr_rv   = NRVHV_FETCH("Errstr");
    s[++i]= c.Handlers_rv = NRVHV_FETCH("Handlers");
    s[++i]= c.Debug_rv    = NRVHV_FETCH("Debug");
    s[++i]= c.imp_datarv  = newRV(imp_datasv);
    s[++i]= c.parent      = (parent) ? newRV(parent) : &sv_undef;

    /* setup other values which don't need to be freed */
    c.version = 2;
    c.h = h;	/* take a copy of the pointer, not a new reference */
    c.imp_dataptr = (void*)SvIV(imp_datasv); /* handy for C pointers */
    c.imp_stash   = imp_stash; /* don't bother taking a ref for this */

    s[0] = newSVpv((char*)&c, sizeof(dbih_com_t));

    /* add our DBI magic to carry the internal handle attributes   */
    sv_magic(SvRV(h), newRV((SV*)av_make(i, s)), DBI_MAGIC, Nullch, 0);

    if (dbis->debug){ /* temporary paranoia: check a random variable matches */
	D_dbihcom(h);
	if (c.Debug_rv != dbihcom->Debug_rv){
	    warn("panic dbih_setup_handle: %p != %p\n", c.Debug_rv, dbihcom->Debug_rv);
	    abort();
	}
    }
}


/* ----------------------------------------------------------------- */
/* Functions implementing Event Handling.                            */

static SV *
dbih_event(dbihcom, name, a1, a2)
    dbih_com_t *dbihcom;
    char *name;
    SV *a1, *a2;
{
    /* We arrive here via DBIh_EVENT* macros (see DBIXS.h) called from	*/
    /* DBD driver C code or $h->event() method (in DBD::_::common)	*/
    AV *handlers_av;
    I32 handlers = 0;
    warn("EVENT '%s', %s, %s", name, neatsvpv(a1), neatsvpv(a2));

    handlers_av = (AV*)SvRV(SvRV(dbihcom->Handlers_rv));
    if (SvTYPE(handlers_av) != SVt_PVAV){
	warn("%s->{'Handlers'} (%s) is not an array reference %d",
		neatsvpv(dbihcom->h), neatsvpv(handlers_av), SvTYPE(handlers_av));
    } else {
	I32 i = handlers = av_len(handlers_av) + 1;
warn("EVENT '%s' has %d handlers", name, i);
	while(--i >= 0){
	    SV *sv = *av_fetch(handlers_av, i, 1);
	    warn("EVENT '%s' handler %d = %s", name, i, neatsvpv(sv));
	    /* call handler here */
	}
    }
    /* if event has still not been handled, call the default handler */
    return &sv_undef;
}


/* ----------------------------------------------------------------- */
/* Functions implementing DBI dispatcher shortcuts.                  */

/* This function implements the DBI FETCH shortcut mechanism.
Any handle attribute FETCH will come to this function (see dispatch).
This function returns either an SV for the fetched value or NULL.
If NULL is returned the dispatcher will call the full FETCH method.
 - If key =~ /^_/ then return NULL (so driver can hide private attribs)
 - If the key does not exist return NULL (may be a virtual attribute).
 - If value is not a ref then return value (the main shortcut).
 - If it's a CODE ref the run CODE and return it's result value!
     (actually it sets a flag so dispatch will run code for us).
 - If it's a ref to a CODE ref then return the CODE ref
     (an escape mechanism to allow real code refs to be stored).
 - Else return NULL (it's some other form of ref, let driver do it).
*/

static SV * 
quick_FETCH(hrv, keysv, imp_msv)
    SV *hrv;	/* ref to inner hash */
    SV *keysv;
    GV **imp_msv;
{
    void *tmp;
    SV *sv;
    STRLEN lp;
    char *key = SvPV(keysv,lp);
    int type;
    if (*key == '_')
	return NULL;	/* never quick_FETCH a 'private' attribute */
    if ( (tmp = hv_fetch((HV*)SvRV(hrv), key, lp, 0)) == NULL)
	return NULL;	/* undefined */
    sv = *(SV**)tmp;
    if (!SvROK(sv))	/* return all non-refs directly	*/
	return sv;	/* this is the main shortcut	*/
    if ( (type=SvTYPE(SvRV(sv))) == SVt_RV
	&& SvTYPE(SvRV(SvRV(sv))) == SVt_PVCV)
	return SvRV(sv); /* return deref if ref to CODE ref */
    if (type != SVt_PVCV)
	return sv;	 /* return non-code refs */
    *imp_msv = SvRV(sv); /* tell dispatch() to execute this code instead */
    return NULL;
}


/* ----------------------------------------------------------------- */
/* ---   The DBI dispatcher. The heart of the perl DBI.          --- */

XS(XS_DBI_dispatch)         /* prototype must match XS produced code */
{
    dXSARGS;

    SV *h = ST(0);          /* the handle we are working with */
    SV *orig_h = h;
    int gimme = GIMME;

    /* XXX This is naughty. We read beyond the end of the stack.    */
    /* Is there a better way? (((XPVCV*)SvANY(cv))->xpv_pv is null) */
    char *meth_name = GvNAME(ST(items));
    AV   *ima_av = (AV*)CvXSUBANY(cv).any_ptr; /* Internal Method Attribs */
    dbih_com_t *dbihcom = NULL;
    HV   *imp_stash = NULL; /* handle implementors stash             */
    SV   *imp_msv   = NULL; /* handle implementors method (GV or CV) */
    SV   *qsv = NULL;       /* quick result from a shortcut method   */

    MAGIC *mg;
    int i, outitems;
    int debug = dbis->debug;  /* local, may change during dispatch    */

    if (debug >= 2){
        fprintf(dbis->log,"    >> %-11s DISPATCH (%s @:%d g:%lx a:%lx)\n",
                meth_name, neatsvpv(h), items, gimme, ima_av);
    }

    if (!SvROK(h) || SvTYPE(SvRV(h)) != SVt_PVHV){
        croak("%s: handle %s is not a hash reference",meth_name,SvPV(h,na));
	/* This will also catch: CLASS->method(); we might want to do */
	/* something better in that case. */
    }

    if (ima_av){	/* Check Internal Method Attributes */
	char *err = NULL;
	char msg[200];

	/* Perform usage checks (will be further optimised later *) */
	int min = SvIV(IMA_FETCH(ima_av, IMA_MINARGS));
	int max = SvIV(IMA_FETCH(ima_av, IMA_MAXARGS));
	if (min && (items < min || (max>0 && items > max))){
	    /* the error reporting is a little tacky here */
	    sprintf(msg, "DBI %s: invalid number of parameters (handle+%d)\n",
			meth_name, items-1);
	    err = msg;
	}
	/* arg type checking could be added here later */
	if (err){
	    char *use = SvPV(IMA_FETCH(ima_av, IMA_ARGS), na);
	    croak("%sUsage: %s->%s(%s)", err, "$h", meth_name, use);
	}
    }

    /* If h is a tied hash ref, switch to the inner ref 'behind' the tie.
       This means all DBI methods work with the inner (non-tied) ref.
       This makes it much easier for methods to access the real hash
       data (without having to go through FETCH and STORE methods) and
       for tie and non-tie methods to call each other.
    */
    if (SvRMAGICAL(SvRV(h)) && (mg=mg_find(SvRV(h),'P'))!=NULL){

        if (SvPVX(mg->mg_obj)==NULL){  /* maybe global destruction */
            if (debug >= 2)
                fprintf(dbis->log,"       (inner handle already deleted)\n");
            XSRETURN_UNDEF;
        }
        /* Distinguish DESTROY of tie (outer) from DESTROY of inner ref. */
	/* This will be used to manually destroy extra internal refs if  */
	/* the application ceases to use the handle.                     */
	/* XXX One day Larry might adopt this concept as an UNTIE method */
        if (*meth_name=='D' && strEQ(meth_name,"DESTROY")){
	    meth_name = "_untied";
        }
        h = mg->mg_obj; /* switch h to inner ref */
        ST(0) = h;      /* switch handle on stack to inner ref */

    }

    /* record this inner handle for use by DBI::var::FETCH */
    /* we use devious means here */
    if (*meth_name=='D' && strEQ(meth_name,"DESTROY")){
	if (SvRVx(dbi_last_h)==SvRV(h))
	    SvRVx(dbi_last_h) = &sv_undef; /* if destroying _this_ handle */
	/* otherwise don't alter it */
    }
    else SvRVx(dbi_last_h) = SvRV(h);

    dbihcom = DBIh_COM(h); /* get common Internal Handle Attributes */

    if ( (i = SvIV(SvRV(dbihcom->Debug_rv))) > debug)
	debug = i;		/* bump up debugging if handle wants it	*/

    /* Now check if we can provide a shortcut implementation here. */
    /* At the moment we only offer a quick fetch mechanism.        */
    if (*meth_name=='F' && strEQ(meth_name,"FETCH")){
	qsv = quick_FETCH(h, ST(1), &imp_msv);
    }

    if (qsv){ /* skip real method call if we already have a 'quick' value */

	ST(0) = sv_mortalcopy(qsv);
	outitems = 1;

    }else{
	if (!imp_msv && (imp_msv = gv_fetchmethod(dbihcom->imp_stash,meth_name)) == NULL)
	    croak("Can't locate DBI object method \"%s\" via package \"%s\"",
		meth_name, HvNAME(dbihcom->imp_stash));

	dbihcom->last_method = meth_name;

	if (debug >= 2){
	    /* Full pkg method name (or just meth_name for ANON CODE) */
	    char *imp_meth_name = (isGV(imp_msv)) ? SvPV(imp_msv,na) : meth_name;
	    /* It would be better to only print HvNAME(imp_stash) if imp_meth_name */
	    /* was in a different class. A job for later. */
	    fprintf(dbis->log,"    -> %s ((%s)%s", imp_meth_name,
			HvNAME(dbihcom->imp_stash), SvPV(orig_h,na));
	    for(i=1; i<items; ++i)
		fprintf(dbis->log," %s", neatsvpv(ST(i)));
	    fprintf(dbis->log, ")\n");
	}

	PUSHMARK(mark);  /* mark our arguments again so we can pass them on	*/

	/* Note that the handle on the stack is still an object blessed	into a	*/
	/* DBI::* class and *not* the DBD::*::* class whose method is being	*/
	/* invoked. This *is* correct and should be largely transparent.	*/

	outitems = perl_call_sv(imp_msv, gimme);

	/* We might perform some fancy error handling here one day (retries etc) */
    }

    if (debug >= 2){
	/* XXX restore local vars so ST(n) works again */
	SPAGAIN; sp -= outitems; ax = (sp - stack_base) + 1; 
	fprintf(dbis->log,"    <- %s=", meth_name);
	if (gimme & G_ARRAY)
	    fprintf(dbis->log," (");
	for(i=0; i < outitems; ++i)
	    fprintf(dbis->log, " %s",  neatsvpv(ST(i)));
	if (gimme & G_ARRAY)
	    fprintf(dbis->log," ) [%d items]", outitems);
	fprintf(dbis->log,"\n");
    }

    XSRETURN(outitems);
}



/* ----------------------------------------------------------------- */
/* The DBI Perl interface (via XS) starts here. Currently these are  */
/* all internal support functions. Note _add_dispatch and see DBI.pm */

MODULE = DBI   PACKAGE = DBI

BOOT:
    dbi_bootinit();

void
_setup_handle(sv, imp_class, parent, imp_datasv=newSViv(0))
    SV *	sv
    char *	imp_class
    SV *	parent
    SV *	imp_datasv
    CODE:
    dbih_setup_handle(sv, imp_class, parent, imp_datasv);
    XSRETURN_YES;

void
_get_imp_data(sv)
    SV *	sv
    CODE:
    {
    D_dbihcom(sv);
    XSRETURN_SV(SvRV(dbihcom->imp_datarv));
    }

void
set_err(sv, errval)
    SV *	sv
    SV *	errval
    CODE:
    {
    D_dbihcom(sv);
    sv_setsv(SvRV(dbihcom->Err_rv), errval);
    XSRETURN_UNDEF;
    }

void
_add_dispatch(meth_name, file, attribs=newSV(0))
    char *	meth_name
    char *	file
    SV *	attribs
    CODE:
    {
    /* install another method name/interface for the dispatcher */
    CV *cv = newXS(meth_name, XS_DBI_dispatch, file);
    if (dbis->debug >= 2)
	fprintf(dbis->log,"_add_dispatch %s,\t%s\n", meth_name, file);

    if (SvROK(attribs) && SvTYPE(SvRV(attribs)) == SVt_PVAV){
	SvREFCNT_inc(SvRV(attribs)); /* ensure it stays around */
	CvXSUBANY(cv).any_ptr = SvRV(attribs);
    }else if (SvOK(attribs)){
	warn("_add_dispatch %s: attributes ignored (not an array ref)",
		meth_name);
    }
    XSRETURN_YES;
    }

int
_debug_dispatch(sv, level=dbis->debug)
    SV *	sv
    int	level
    CODE:
    /* Return old/current value. No change if new value not given */
    if (level != dbis->debug)
	fprintf(dbis->log,"    DBI dispatch debug level set to %d\n", level);
    RETVAL = dbis->debug;
    dbis->debug = level;
    sv_setiv(perl_get_sv("DBI::dbi_debug",0x5), dbis->debug);
    if (dbis->debug >= 2)
	perl_destruct_level = 2;
    OUTPUT:
    RETVAL


int
_debug_handle(sv, level=0)
    SV *	sv
    int	level
    CODE:
    {
    D_dbihcom(sv);
    SV *dsv = SvRV(dbihcom->Debug_rv);
    /* Return old/current value. No change if new value not given */
    RETVAL=SvIV(dsv);
    if (items == 2 && level != RETVAL){ /* set value */
	sv_setiv(dsv, level);
	fprintf(dbis->log,"    %s debug level set to %d\n", SvPV(sv,na), level);
    }
    }
    OUTPUT:
    RETVAL


void
_svdump(sv)
    SV *	sv
    CODE:
    fprintf(dbis->log, "DBI::svdump(%s)", SvPV(ST(0),na));
    sv_dump(ST(0));


MODULE = DBI   PACKAGE = DBI::var

void
FETCH(sv)
    SV *	sv
    PPCODE:
    /* Note that we do not come through the dispatcher to get here.	*/
    char *meth = SvPV(SvRV(sv),na);	/* what should this tie do ?	*/
    char type = *meth++;		/* is this a $ or & style	*/
    GV *imp_gv;
    SV *result;
    SV *lhp = dbi_last_h;
    int ok = ( SvRV(lhp) != &sv_undef );

    D_dbihcom(dbi_last_h);

    if (dbis->debug >= 2){
	SvROK_on(dbi_last_h);
	fprintf(dbis->log,"    <> $DBI::%s (%c) FETCH from lasth=%s\n",
		meth, type, (ok)?SvPV(lhp,na):"none");
	SvROK_off(dbi_last_h);
    }

    if (type == '!'){	/* special case for $DBI::lasth */
	/* for the time being we don't return the actual handle */
	if (ok){ XSRETURN_YES; } else { XSRETURN_NO; }
    }
    if ( !ok ){		/* warn() may be changed to a debug later */
	warn("Can't read $DBI::%s, no last handle", meth);
	XSRETURN_UNDEF;
    }
    if (type == '*'){	/* special case for $DBI::err	*/
	SV *errsv = dbihcom->Err_rv;
	if (dbis->debug >= 2)
	    fprintf(dbis->log,"err = '%s'\n", neatsvpv(SvRV(errsv)));
	XSRETURN_SV(SvRV(errsv));
    }
    if (type == '$'){ /* lookup scalar variable in implementors stash */
	char *vname = mkvname(dbihcom->imp_stash, meth, 0);
	SV *vsv = perl_get_sv(vname, 1);
	if (dbis->debug >= 2)
	    fprintf(dbis->log,"%s = %s\n", vname, neatsvpv(vsv));
	XSRETURN_SV(vsv);
    }
    /* default to method call via stash of implementor of dbi_last_h */
    if (dbis->debug >= 2)
	fprintf(dbis->log,"%s::%s\n", HvNAME(dbihcom->imp_stash), meth);
    ST(0) = lhp;
    if ((imp_gv = gv_fetchmethod(dbihcom->imp_stash,meth)) == NULL){
	warn("Can't locate $DBI::%s object method \"%s\" via package \"%s\"",
	    meth, meth, HvNAME(dbihcom->imp_stash));
	XSRETURN_UNDEF;
    }
/* something here is not quite right ! (wrong number of args to method for example) */
    PUSHMARK(mark);  /* reset mark (implies one arg as we were called with one arg?) */
    perl_call_sv(imp_gv, GIMME);
    XSRETURN(1);


MODULE = DBI   PACKAGE = DBD::_::common

void
event(h, type, a1=&sv_undef, a2=&sv_undef)
    SV *	h
    char *	type
    SV *	a1
    SV *	a2
    PPCODE:
    {
    D_dbihcom(h);
    XSRETURN_SV(DBIh_EVENT2(type, a1, a2));
    }

void
private_data(h)
    SV *	h
    PPCODE:
    {
    D_dbihcom(h);
    XSRETURN_SV(SvRV(DBIHCOM->imp_datarv));
    }


# end
